module typist


//	**************************************************************************************************
//
//	This program creates two interactive processes:
//	-	One process presents a simple text window in which text can be typed (this module).
//	-	The other process keeps track of the number of typed keys per minute (monitor module).
//	Communication is done by means of message passing.
//	In a future distributed version the two processes can be run on different processors.
//
//	The program has been written in Clean 1.3.1 and uses the Clean Standard Object I/O library 1.1
//	
//	**************************************************************************************************


import StdEnv, StdIO
import monitor


::	Typist							// The local process state
	=	{	firstKeyTyped	:: Bool	// The first key is typed in the typist window
		}


NrOfLines			:== 10
NrOfMaxCharsPerLine	:== 30


Start :: *World -> *World
Start world
	#	(font,    world)	= accScreenPicture openDialogFont world
	#	(metrics, world)	= accScreenPicture (getFontMetrics font) world
	#	(rId,     world)	= openRId world
	#	(ids,     world)	= openIds 5 world
	=	startProcesses (ProcessGroup 0 (typistProcess font metrics rId ids)) world

typistProcess :: Font FontMetrics (RId MonitorMessage) [Id] -> SDIProcess (Window EditControl) .p
typistProcess font metrics rId ids=:[wId,editId,mId,runId,tId]
	=	SDIProcess {firstKeyTyped=False} undef window [initIO] []
where
//	initIO initialises the typist process.
	initIO typist
		#	(error,typist)		= openMenu undef menu typist
		|	error<>NoError
			=	abort "Typist could not open menu."
		#	(error,typist)		= openTimer undef timer typist
		|	error<>NoError
			=	abort "Typist could not open timer."
		#	(Just pos,typist)	= accPIO (getWindowPos wId) typist
			monitorpos			= (LeftTop,{pos & vy=pos.vy+wSize.h})
		#	typist				= openMonitor monitorpos rId typist
		|	otherwise
			=	typist

//	window is the single document of the typist process. 
//	Keyboard information is sent to the monitor process.
	window	= Window "Typist window" 
				(	EditControl "" (wSize.w-2*metrics.fMaxWidth) NrOfLines
				[	ControlKeyboard		keyFilter Able (noLS1 sendKeys)
				,	ControlId			editId
				,	ControlSelectState	Unable
				]
				)
				[	WindowId			wId
				,	WindowSize			wSize
				]
	where
//		Filter only non-repeating character keys to the monitor process.
		keyFilter (CharKey _ (KeyDown repeat))	= not repeat
		keyFilter _								= False
		
//		Key messages start with BeginSession so that monitoring will start after the first key hit.
//		Only after the first key hit the timer stopwatch is activated.
		sendKeys :: KeyboardState (PSt Typist .p) -> PSt Typist .p
		sendKeys keyboard typist=:{ls=local=:{firstKeyTyped}}
			|	firstKeyTyped
				#	(_,typist)	= syncSend rId (KeyHit char) typist
				=	typist
			|	otherwise
				#	local		= {local & firstKeyTyped=True}
				#	typist		= {typist & ls=local}
				#	(_,typist)	= syncSend rId BeginSession typist
				#	typist		= appPIO (enableTimer tId) typist
				#	(_,typist)	= syncSend rId (KeyHit char) typist
				=	typist
		where
			(IsCharKey char)	= getKeyboardStateKey keyboard
	
	wSize	= {	w	= (NrOfMaxCharsPerLine+2)*metrics.fMaxWidth
			  ,	h	= (NrOfLines+6)*(fontLineHeight metrics)
			  }
	
//	menu defines the commands of the typist process. 
	menu	= Menu "File"
				(	MenuItem "Run"  [MenuShortKey 'r', MenuFunction (noLS run), MenuId runId]
				:+:	MenuSeparator	[]
				:+:	MenuItem "Quit" [MenuShortKey 'q', MenuFunction (noLS closeProcess)]
				)
				[	MenuId	mId
				]
	where
//		run starts a session by enabling the edit control that receives the keyboard input.
		run :: (PSt Typist .p) -> PSt Typist .p
		run typist=:{ls,io}
			#	io	= setWindow wId [enableControls [editId],setControlTexts [(editId,"")]] io
			#	io	= setMenu   mId [disableMenuElements [runId]] io
			=	{typist & ls={ls & firstKeyTyped=False},io=io}

//	timer will end a typing session after 60 seconds. The timer is enabled by sendKeys.	
	timer	= Timer (60*ticksPerSecond) NilLS
				[	TimerId				tId
				,	TimerSelectState	Unable
				,	TimerFunction		(noLS1 endOfSession)
				]
	where
//		The monitor process is notified of the end of the session by receiving the EndSession message.
		endOfSession :: NrOfIntervals (PSt Typist .p) -> PSt Typist .p
		endOfSession _ typist=:{io}
			#	io			= disableTimer tId io
			#	io			= setWindow wId [disableControls [editId]] io
			#	io			= setMenu   mId [enableMenuElements [runId]] io
			#	(_,typist)	= asyncSend rId EndSession {typist & io=io}
			=	typist
